import React, { useEffect, useRef, useState } from "react";
import { useSelector } from "react-redux";
import { RootState } from "@/src/store";

interface AuctionTimerProps {
  onAuctionEnd?: () => void;
  useParentProps?: boolean;        // ✅ keep this
  className?: string;
  auctionEndsAt?: string;          // ISO string (optional)
  currentServerTime?: string;      // ISO string (optional)
}

const pad = (n: number) => n.toString().padStart(2, "0");

const AuctionTimer: React.FC<AuctionTimerProps> = ({
  onAuctionEnd,
  useParentProps = true,
  auctionEndsAt,
  currentServerTime,
  className = "",
}) => {
  const { data: listingData } = useSelector(
    (state: RootState) => state.viewListingReducer
  );
  const viewListingData = listingData?.data?.listing;

  // Decide which props to use
  const serverTime = useParentProps
    ? currentServerTime
    : viewListingData?.current_time_formatted;

  const auctionEndTime = useParentProps
    ? auctionEndsAt
    : viewListingData?.auction_ends_at;

  const [display, setDisplay] = useState("Loading...");
  const [ended, setEnded] = useState(false);

  const onAuctionEndRef = useRef(onAuctionEnd);
  useEffect(() => { onAuctionEndRef.current = onAuctionEnd; }, [onAuctionEnd]);

  useEffect(() => {
    if (!auctionEndTime || !serverTime) return;

    // Parse ISO string to milliseconds
    const parseTime = (isoString: string) => new Date(isoString).getTime();

    const endMs = parseTime(auctionEndTime);
    let nowMs = parseTime(serverTime);

    const updateTimer = () => {
      const distance = endMs - nowMs;

      if (distance <= 0) {
        setDisplay("00h : 00m : 00s");
        setEnded(true);
        if (onAuctionEndRef.current) onAuctionEndRef.current();
        return;
      }

      const totalSeconds = Math.floor(distance / 1000);
      const totalHours = Math.floor(totalSeconds / 3600); // total hours
      const minutes = Math.floor((totalSeconds % 3600) / 60);
      const seconds = totalSeconds % 60;

      setDisplay(`${totalHours}h : ${pad(minutes)}m : ${pad(seconds)}s`);

      nowMs += 1000; // increment simulated server time
    };

    updateTimer(); // initial call
    const interval = setInterval(updateTimer, 1000);
    return () => clearInterval(interval);
  }, [auctionEndTime, serverTime]);

  return (
    <div className={`auction-timer ${className} ${ended ? "ended" : ""}`}>
      {display}
    </div>
  );
};

export default React.memo(AuctionTimer);

