import React, { useState } from "react";

interface AddressCardProps {
  fullname: string;
  address?: string;
  province: string;
  city: string;
  state?: string;
  postal_code: string;
  country: string;
  house_number: string;
  location: string;
  onEdit: () => void;
  onDelete: () => void;
  isLast: boolean;
}

const AddressCard: React.FC<AddressCardProps> = ({
  fullname,
  province,
  city,
  location,
  house_number,
  country,
  state,
  postal_code,
  onEdit,
  onDelete,
  isLast,
}) => {
  const [showDeleteConfirm, setShowDeleteConfirm] = useState(false);

  const handleDeleteClick = () => {
    setShowDeleteConfirm(true);
  };

  const confirmDelete = () => {
    onDelete();
    setShowDeleteConfirm(false);
  };

  const cancelDelete = () => {
    setShowDeleteConfirm(false);
  };

  return (
    <div className="bg-white rounded-md p-2 mt-3 lg:w-[437px] lg:min-h-[250px] relative ">
      {/* {/ Delete Confirmation Popup /} */}
      {showDeleteConfirm && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white p-6 rounded-lg shadow-xl max-w-sm w-full">
            <h3 className="text-xl font-bold mb-4 text-black">
              Confirm Deletion
            </h3>
            <p className="mb-6 text-gray-600">
              Are you sure you want to delete this address?
            </p>
            <div className="flex justify-center space-x-4">
              <button
                onClick={cancelDelete}
                className="px-4 py-2 border text-gray-600 border-gray-300 rounded-md hover:bg-gray-100 transition-colors"
              >
                Cancel
              </button>
              <button
                onClick={confirmDelete}
                className="px-4 py-2 bg-red-600 text-white rounded-md hover:bg-red-700 transition-colors"
              >
                Delete
              </button>
            </div>
          </div>
        </div>
      )}

      <div className="text-center mt-4 flex flex-col pb-6 ">
        <h2 className="lg:text-xl font-bold text-start text-[#013DCB] capitalize">
          {fullname}
        </h2>
        <p className="lg:text-xl text-start font-semibold text-[#262626] mt-2 md:h-[126px] overflow-auto">
          {`${house_number}`} {`${city}`} {`${country}`} {`${postal_code}`}{" "}
          {province && `${province}`} {state && `${state}`}
        </p>
        <p className="lg:text-xl text-start font-semibold text-[#262626] ">
          {location}
        </p>

        <div className="flex items-center lg:gap-14 xs:gap-11 pt-9">
          <p
            className="lg:text-xl text-[#A2A2A1] font-semibold cursor-pointer hover:text-black"
            onClick={onEdit}
          >
            Edit
          </p>
          <p className="text-[#A2A2A1]">|</p>
          <p
            className="lg:text-xl text-[#A2A2A1] font-semibold cursor-pointer hover:text-black"
            onClick={handleDeleteClick}
          >
            Remove
          </p>
        </div>
      </div>
      {!isLast && (
        <div className="absolute left-5 border-b-2 bottom-0 w-[90%] lg:top-14 lg:bottom-14 lg:left-full lg:border-r-2 lg:w-0 border-gray-200"></div>
      )}
    </div>
  );
};

export default AddressCard;
