"use client";

import React, { useEffect, useState } from "react";
import dynamic from "next/dynamic";
import { Formik, Form, FormikHelpers } from "formik";
import * as Yup from "yup";

// Dynamically import OTPInput with SSR disabled
const OTPInput = dynamic(
  () => import("react-otp-input").then((mod) => mod.default),
  {
    ssr: false,
  }
);

interface FormValues {
  otp: string;
}

const OTP: React.FC = () => {
  const [otp, setOtp] = useState("");
  const [inputWidth, setInputWidth] = useState("3rem");
  useEffect(() => {
    const handleResize = () => {
      if (window.innerWidth < 370) {
        setInputWidth("2rem");
      } else {
        setInputWidth("3rem");
      }
    };

    window.addEventListener("resize", handleResize);
    handleResize();

    return () => window.removeEventListener("resize", handleResize);
  }, []);

  const handleChange = (
    otpValue: string,
    setFieldValue: (field: string, value: any, shouldValidate?: boolean) => void
  ) => {
    setOtp(otpValue);
    setFieldValue("otp", otpValue);
  };

  return (
    <div className="max-w-md mx-auto bg-white mt-10 rounded-xl p-8 px-10 mb-10">
      <h1 className="text-3xl font-bold mb-4">Enter OTP</h1>
      <p className="text-sm font-semibold text-center mb-10">
        We have sent an OTP on your Email/Mobile number. Please enter OTP here.
      </p>

      <Formik
        initialValues={{ otp: "" }}
        validationSchema={Yup.object({
          otp: Yup.string()
            .length(4, "OTP must be exactly 4 digits")
            .required("OTP is required"),
        })}
        onSubmit={(
          values: FormValues,
          { setSubmitting }: FormikHelpers<FormValues>
        ) => {
          console.log("Submitted OTP:", values);
          setSubmitting(false);
        }}
      >
        {(formikProps) => {
          const { setFieldValue, errors, touched } = formikProps;
          return (
            <Form className="space-y-4">
              <div>
                <OTPInput
                  value={otp}
                  onChange={(otpValue: string) => handleChange(otpValue, setFieldValue)}
                  numInputs={4}
                  containerStyle={{ justifyContent: "space-evenly" }}
                  inputStyle={{
                    borderRadius: "5px",
                    width: inputWidth,
                    height: "3rem",
                    fontSize: "1.5rem",
                    border: "1px solid #ccc",
                  }}
                  focusStyle={{
                    border: "1px solid #3b82f6",
                    outline: "none",
                  }}
                />
                {/* <OTPInput
                  value={otp}
                  onChange={(otpValue: string) =>
                    handleChange(otpValue, setFieldValue)
                  }
                  numInputs={4}
                  containerStyle={{ justifyContent: "space-evenly" }}
                  renderInput={(props) => (
                    <input
                      {...props}
                      className="border rounded-md text-center text-xl w-12 h-12 focus:outline-none focus:border-blue-500"
                    />
                  )}
                /> */}

                {touched.otp && errors.otp && (
                  <div className="text-red-500 mt-2">{errors.otp}</div>
                )}
              </div>

              <div>
                <button
                  type="submit"
                  className="text-white bg-blue-600 py-2 px-4 rounded-md w-full"
                >
                  Verify
                </button>
                <p className="text-sm font-semibold text-end text-blue-500 cursor-pointer mt-2">
                  Resend OTP
                </p>
              </div>
            </Form>
          );
        }}
      </Formik>
    </div>
  );
};

export default OTP;
