// ImagePreviewModal.tsx
import React, { useRef } from "react";
import Slider from "react-slick";
import { SlArrowLeft, SlArrowRight } from "react-icons/sl";
import Image from "next/image";

interface ImagePreviewModalProps {
  images: string[];
  selectedIndex: number;
  isOpen: boolean;
  onClose: () => void;
}

const ImagePreviewModal: React.FC<ImagePreviewModalProps> = ({
  images,
  selectedIndex,
  isOpen,
  onClose,
}) => {
  const sliderRef = useRef<Slider | null>(null);
  const thumbSliderRef = useRef<Slider | null>(null);

  if (!isOpen) return null;

  return (
    <div
      className="fixed inset-0 bg-black bg-opacity-90 flex justify-center items-center z-50"
      onClick={onClose}
    >
      <div className="relative bg-black p-4 flex flex-col rounded-lg w-full h-full">
        {/* Close Button */}
        <button
          className="absolute top-4 right-4 text-white text-2xl"
          onClick={onClose}
        >
          ✕
        </button>

        {/* Main Image Slider */}
        <div className="flex-1 flex justify-center items-center">
          <Slider
            ref={(slider) => {
              sliderRef.current = slider;
            }}
            dots={false}
            infinite={true}
            speed={500}
            slidesToShow={1}
            slidesToScroll={1}
            arrows={false}
            initialSlide={selectedIndex}
            className="w-full max-w-4xl md:h-[70vh]"
          >
            {images.map((image, index) => (
              <div key={index} className="flex justify-center">
                <Image
                  unoptimized
                  src={image}
                  className="w-full md:h-[70vh] rounded-lg"
                  alt="Slide"
                />
              </div>
            ))}
          </Slider>
        </div>

        {/* Navigation Buttons */}
        <button
          className="absolute left-6 top-1/2 transform -translate-y-1/2 bg-gray-200 p-2 rounded-full shadow-lg hover:bg-yellow-300"
          onClick={(e) => {
            e.stopPropagation();
            sliderRef.current?.slickPrev();
          }}
        >
          <SlArrowLeft />
        </button>
        <button
          className="absolute right-4 top-1/2 transform -translate-y-1/2 bg-gray-200 p-2 rounded-full shadow-lg hover:bg-yellow-300"
          onClick={(e) => {
            e.stopPropagation();
            sliderRef.current?.slickNext();
          }}
        >
          <SlArrowRight />
        </button>

        {/* Thumbnail Images */}
        <div className="w-full max-w-7xl mx-auto mt-4">
          <Slider
            ref={(thumbSlider) => {
              thumbSliderRef.current = thumbSlider;
            }}
            dots={false}
            infinite={images.length > 4}
            speed={500}
            slidesToShow={
              images.length < 4 ? images.length : Math.min(7, images.length)
            }
            slidesToScroll={1}
            arrows={images.length > 1}
          >
            {images.map((image, index) => (
              <div key={index} className="px-1">
                <Image
                  unoptimized
                  src={image}
                  className="md:h-32 xs:h-24 object-cover cursor-pointer rounded-md"
                  alt="Thumbnail"
                  onClick={(e) => {
                    e.stopPropagation();
                    sliderRef.current?.slickGoTo(index);
                  }}
                />
              </div>
            ))}
          </Slider>
        </div>
      </div>
    </div>
  );
};

export default ImagePreviewModal;
