"use client";

import CashOut from "../../../../assets/walletCash.svg";
import React, { useEffect, useState } from "react";
import usaFlag from "../../../../assets/united-states-flag-icon.svg";
import canFlag from "../../../../assets/canada-flag-icon.svg";
import Image from "next/image"; // Make sure this is imported
import { useRouter } from "next/navigation";

// Define the props type
interface ConfirmCashInModalProps {
  onClose: () => void;
}

const currencyRates: Record<string, { symbol: string; flag: string }> = {
  usd: { symbol: "usd $", flag: usaFlag },
  cad: { symbol: "cad $", flag: canFlag },
};

const ConfirmCashInModal: React.FC<ConfirmCashInModalProps> = ({ onClose }) => {
  const [amountToAdd, setAmountToAdd] = useState(0);
  const [currency, setCurrency] = useState("usd");
  const router = useRouter(); // ✅ useRouter from next/navigation

  useEffect(() => {
    if (amountToAdd > 0) {
      router.prefetch("/profile/wallet/payment");
    }
  }, [amountToAdd, router]);

  const handleAddAmount = () => {
    if (amountToAdd > 0) {
      // ✅ Use query parameters instead of React Router `state`
      router.push(
        `/profile/wallet/payment?walletAmount=${amountToAdd}&currency=${currency}`
      );
      setAmountToAdd(0);
    } else {
      alert("Please enter a valid amount to add.");
    }
  };

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
      <div className="bg-white w-[90%] max-w-md rounded-lg shadow-lg p-6 relative">
        <button
          onClick={onClose}
          className="absolute top-0 right-3 text-black font-normal cursor-pointer hover:text-gray-800 text-4xl p-0"
        >
          &times;
        </button>
        <div className="text-center mb-6">
          <Image
            unoptimized
            src={CashOut}
            alt="Cash Out Icon"
            width={64} // Equivalent to w-16
            height={64} // Equivalent to h-16
            className="mx-auto mb-4 w-16 h-16"
          />
          <h2 className="text-2xl font-semibold text-[#151D48] font-sans">
            Confirm Cash In
          </h2>
          <p className="text-gray-500 text-sm">
            Please confirm your cash-in details. Money will be added to your
            account.
          </p>
        </div>
        <div className="space-y-4">
          <div className="flex flex-col items-center text-gray-700">
            <span className="font-sans pb-2 text-lg font-medium">
              Add Amount
            </span>
            <div className="relative w-[87%] ">
              <select
                value={currency}
                onChange={(e) => setCurrency(e.target.value)}
                className="border border-gray-300 rounded-lg px-3 py-2 w-[100%] shadow-sm text-lg mb-5"
              >
                {Object.keys(currencyRates).map((curr) => (
                  <option key={curr} value={curr} className="capitalize">
                    {currencyRates[curr].symbol.toUpperCase()}
                  </option>
                ))}
              </select>
              <Image
                unoptimized
                src={currencyRates[currency].flag}
                alt={currency}
                width={24}
                height={24}
                className="absolute right-8 top-3 w-6 h-6"
              />
            </div>
            <div className="flex items-center border border-gray-300 rounded-lg w-[87%] px-4 py-2 mb-4 shadow-sm">
              <Image
                unoptimized
                src={currencyRates[currency].flag}
                alt={currency}
                width={24} // w-6 = 24px
                height={24} // h-6 = 24px
                className="mr-2"
              />
              <span className="mr-2 text-lg font-medium gap-1">
                {currencyRates[currency].symbol.toUpperCase()}
              </span>
              <input
                type="number"
                value={amountToAdd}
                onChange={(e) => setAmountToAdd(parseFloat(e.target.value))}
                placeholder="Enter amount"
                className="flex-1 outline-none text-base bg-transparent border border-gray-300 rounded-md w-[90%]"
              />
            </div>
          </div>
        </div>
        <div className="bg-gray-100 p-4 mt-4 rounded-md text-start shadow-sm">
          <p className="text-sm text-black font-medium">
            <strong>About your earnings</strong>
          </p>
          <p className="text-sm text-[#454545]">
            The amount shown is what will be transferred to your account.
            Currency rates are managed within that.
          </p>
        </div>
        <div className="flex justify-between mt-6">
          <button
            onClick={onClose}
            className="px-6 py-2 text-gray-600 font-medium border border-[#04104E] rounded-lg hover:bg-[#04104E] hover:text-white transition-all duration-300 shadow-sm"
          >
            Cancel
          </button>
          <button
            onClick={handleAddAmount}
            className="px-6 py-2 bg-[#04104E] text-white font-medium rounded-lg hover:bg-[#151D48] transition-all duration-300 shadow-sm"
          >
            Confirm
          </button>
        </div>
      </div>
    </div>
  );
};

export default ConfirmCashInModal;
