// src/common/HeroCategorySlider.tsx
"use client";

import Image from "next/image";
import { useRef } from "react";
import roundArrowRight from "../assets/rightarrow.svg";
import roundArrowLeft from "../assets/LeftArrow.svg";
import { useRouter } from "next/navigation";
import dynamic from "next/dynamic";

// Define the type for the Slider methods
type SliderMethods = {
  slickPrev: () => void;
  slickNext: () => void;
};

// Dynamically import Slider as it's a client-side library
const Slider = dynamic(() => import("react-slick").then((mod) => mod.default), {
  ssr: false, // Ensure it's only rendered on the client
  loading: () => <div className="flex space-x-4 overflow-x-auto" />, // Placeholder while loading
}) as React.ComponentType<any>;

// Slider settings (can be moved outside if static)
const sliderSettings = {
  dots: false,
  infinite: true,
  speed: 1000,
  slidesToShow: 6,
  slidesToScroll: 1,
  arrows: false,
  autoplay: true,
  responsive: [
    {
      breakpoint: 1024,
      settings: {
        slidesToShow: 4,
        slidesToScroll: 1,
        infinite: true,
        dots: false,
      },
    },
    { breakpoint: 600, settings: { slidesToShow: 1, slidesToScroll: 1 } },
  ],
};

interface HeroCategorySliderProps {
  categories: any[]; // Expect categories data as a prop from the Server Component
  CategoryBackgroundSrc: string; // Expect the background image src
}

export default function HeroCategorySlider({
  categories,
  CategoryBackgroundSrc,
}: HeroCategorySliderProps) {
  const router = useRouter();
  const sliderRef = useRef<SliderMethods | null>(null);

  const handlePrev = () => sliderRef.current?.slickPrev();
  const handleNext = () => sliderRef.current?.slickNext();

  const getCategoryUrl = (id: number, name: string): string => {
    const categoryName = name
      ?.replace(/[^a-zA-Z0-9\s]/g, "")
      .replace(/\s+/g, "-")
      .toLowerCase();
    return `/${id}/${categoryName}`;
  };

  const handleCategoryClick = (id: number, name: string) => {
    const url = getCategoryUrl(id, name);
    router.push(url);
  };

  const handleCategoryPrefetch = (id: number, name: string) => {
    const url = getCategoryUrl(id, name);
    router.prefetch(url);
  };

  return (
    <div
      className="absolute left-1/2 transform -translate-x-1/2 md:block hidden bottom-[-147px] mx-auto pl-[1.5rem] pr-[1.5rem] lg:max-w-[95vw] w-[80rem] sm:w-[70rem] md:w-[80rem] max-h-[350px] pb-6 font-system font-bold text-black rounded-2xl bg-cover bg-center"
      style={{ backgroundImage: `url(${CategoryBackgroundSrc})` }}
    >
      <div className="text-center text-[32px] text-white flex items-center justify-center">
        <div className="flex-grow border-t border-white-400 mx-4"></div>
        <p className="text-center text-[32px] text-white font-inter-tight">
          Find Auctions By Category
        </p>
        <div className="flex-grow border-t border-white-400 mx-4"></div>
      </div>

      <Slider ref={sliderRef} {...sliderSettings}>
        {categories?.map((category: any) => (
          <div
            key={category.id}
            className="slide-item"
            onClick={() => handleCategoryClick(category.id, category.name)}
            onMouseEnter={() =>
              handleCategoryPrefetch(category.id, category.name)
            }
          >
            <div className="p-2 my-3 ml-[5px] cursor-pointer shadow-lg flex flex-col bg-white justify-start items-center h-40 w-[190px]  rounded-lg">
              <div className="relative w-[120px] h-[100px]">
                <Image
                  unoptimized
                  src={category.icon}
                  alt={category.name}
                  fill
                  sizes="140px"
                  className="object-contain"
                  quality={75}
                  loading="lazy"
                />
              </div>
              <p className="text-base font-semibold text-center">
                {category.name}
              </p>
            </div>
          </div>
        ))}
      </Slider>

      <div className="flex space-x-2">
        <Image
          unoptimized
          src={roundArrowLeft}
          alt="Previous"
          onClick={handlePrev}
          className="hover:-translate-x-1 cursor-pointer w-4 h-8 absolute top-[50%] left-2"
        />
        <Image
          unoptimized
          src={roundArrowRight}
          alt="Next"
          onClick={handleNext}
          className="hover:translate-x-1 cursor-pointer w-4 h-8 absolute top-[50%] right-[0.5rem]"
        />
      </div>
    </div>
  );
}
